<?php

require_once $config->get('sysRoot').'alpha/util/AlphaErrorHandlers.inc';
require_once $config->get('sysRoot').'alpha/util/Logger.inc';
require_once $config->get('sysRoot').'alpha/model/TagObject.inc';
require_once $config->get('sysRoot').'alpha/util/cache/AlphaCacheProviderFactory.inc';

/**
 * A widget for rendering a tag cloud, based off the TagObject instances in the
 * database.
 * 
 * @package alpha::view::widgets
 * @since 1.0
 * @author John Collins <dev@alphaframework.org>
 * @version $Id: TagCloud.inc 1468 2011-12-13 19:52:58Z johnc $
 * @license http://www.opensource.org/licenses/bsd-license.php The BSD License
 * @copyright Copyright (c) 2011, John Collins (founder of Alpha Framework).  
 * All rights reserved.
 * 
 * <pre>
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the 
 * following conditions are met:
 * 
 * * Redistributions of source code must retain the above 
 *   copyright notice, this list of conditions and the 
 *   following disclaimer.
 * * Redistributions in binary form must reproduce the above 
 *   copyright notice, this list of conditions and the 
 *   following disclaimer in the documentation and/or other 
 *   materials provided with the distribution.
 * * Neither the name of the Alpha Framework nor the names 
 *   of its contributors may be used to endorse or promote 
 *   products derived from this software without specific 
 *   prior written permission.
 *   
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND 
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * </pre>
 *  
 */
class TagCloud {
	
	/**
	 * Trace logger
	 * 
	 * @var Logger
	 * @since 1.0
	 */
	private static $logger = null;
	
	/**
	 * A hash array of popular tags
	 * 
	 * @var array
	 * @since 1.0
	 */
	private $popTags = array();
	
	/**
	 * Constructor
	 * 
	 * @param $limit The maximum amount of tags to include in the cloud.
	 * @param $cacheKey Set this optional value to attempt to store the tag cloud array in the available cache for 24hrs (sysCacheProviderName).
	 * @since 1.0
	 */
	public function __construct($limit, $cacheKey = '') {
		self::$logger = new Logger('TagCloud');
		
		global $config;
		
		if($cacheKey != '' && $config->get('sysCacheProviderName') != '') {
			$cache = AlphaCacheProviderFactory::getInstance($config->get('sysCacheProviderName'));
      		$this->popTags = $cache->get($cacheKey);
      		
      		// cache look-up failed, so add it for the next time
      		if(!$this->popTags) {
      			self::$logger->debug('Cache lookup on the key ['.$cacheKey.'] failed, regenerating popular tags...');
      			
      			$this->popTags = TagObject::getPopularTagsArray($limit);
      			
      			$cache->set($cacheKey, $this->popTags, 86400);
      		}else{
      			$this->popTags = array_slice($this->popTags, 0, $limit);
      			self::$logger->debug('Cache lookup on the key ['.$cacheKey.'] succeeded');
      		}
		}else{
			$this->popTags = TagObject::getPopularTagsArray($limit);
		}
	}
	
	/**
	 * Render the tag cloud and return all of the HTML links in a single paragraph.
	 * 
	 * @param $minLinkSize The minimum font size for any tag link, in points.
	 * @param $maxLinkSize The maximum font size for any tag link, in points.
	 * @param $target The target attribute for the links 
	 * @return string
	 * @since 1.0
	 */
	public function render($minLinkSize=8, $maxLinkSize=20, $target='') {
		global $config;
		
		$html = '<p>';
		
		foreach (array_keys($this->popTags) as $key) {
			$linkSize = $this->popTags[$key];
			if ($linkSize < $minLinkSize)
				$linkSize = $minLinkSize;
			if ($linkSize > $maxLinkSize)
				$linkSize = $maxLinkSize;
			$html .= '<a href="'.$config->get('sysURL').'search/q/'.$key.'" style="font-size:'.$linkSize.'pt;"'.(empty($target) ? '' : ' target="'.$target.'"').' rel="tag">'.$key.'</a> ';
		}
		
		return $html.'</p>';
	}
}

?>